/*
 * Created on 24.07.2004
 * Copyright (C) 2004 Sebastian Baltes (sebastian.baltes@gmx.de)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2.1 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 */
package com.ecyrd.jspwiki.plugin;

import java.util.Arrays;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.apache.log4j.Category;

import com.ecyrd.jspwiki.WikiContext;
import com.ecyrd.jspwiki.attachment.Attachment;

/**
 * Plugin that displays latex math syntax as gif images. Parameters may also be
 * read from JspWiki-Properties. In this case the parameter names with prefix
 * "math." are used as property names.
 * 
 * @author Sebastian Baltes
 */
public class Applet implements WikiPlugin {

  private static final Category log = Category.getInstance(Applet.class);

  public static final String PARAM_CODE = "code";

  public static final String PARAM_ALIGN = "align";

  public static final String PARAM_HEIGHT = "height";

  public static final String PARAM_WIDTH = "width";

  public static final String PARAM_ARCHIVE = "archive";

  public static final String PARAM_CODEBASE = "codebase";

  public static final String PARAM_ALT = "alt";

  public static final List ALL_PARAMETERS = Arrays.asList(new String[] { 
      PluginManager.PARAM_BODY, PARAM_CODE, PARAM_ALIGN, PARAM_HEIGHT, 
      PARAM_WIDTH, PARAM_ARCHIVE, PARAM_CODEBASE, PARAM_ALT 
  });

  /*
   * @see com.ecyrd.jspwiki.plugin.WikiPlugin#execute(com.ecyrd.jspwiki.WikiContext,
   *      java.util.Map)
   */
  public final String execute(WikiContext context, Map params) throws PluginException {
    try {
      return executeSafe(context, params);
    } catch (PluginException e) {
      throw e;
    } catch (Exception e) {
      String message = "error in " + this.getClass().getName() + ": " + e.getMessage();
      log.info(message, e);
      throw new PluginException(message);
    }
  }

  /**
   * @param context
   * @param params
   * @return
   */
  protected String executeSafe(WikiContext context, Map params) throws Exception {
    // parsing
    PluginParameterParser parser = new PluginParameterParser(params, "Applet plugin");
    String className = parser.getString(PARAM_CODE);
    String archive = parser.getString(PARAM_ARCHIVE, null);
    String codebase = parser.getString(PARAM_CODEBASE, null);
    Integer width = parser.getInteger(PARAM_WIDTH, new Integer(300));
    Integer height = parser.getInteger(PARAM_HEIGHT, new Integer(300));
    String align = parser.getString(PARAM_ALIGN, null);
    String alt = parser.getString(PARAM_ALT, null);

    // delete common parameters. the result is the list of applet parameters
    params.keySet().removeAll(ALL_PARAMETERS);

    // write html
    StringBuffer result = new StringBuffer();
    result.append("<applet code=\"" + className + "\" ");
    Attachment attachment = null;
    if (archive != null) {
      attachment = context.getEngine().getAttachmentManager().getAttachmentInfo(context, archive);
      if (attachment != null) {
        archive = context.getEngine().getAttachmentURL(attachment.getName());
      }
      result.append(" archive=\"" + archive + "\"");
    }
    if (attachment == null) {
      if (codebase != null) {
        result.append(" codebase=\"" + codebase + "\"");
      }
    }
    if (align != null) {
      result.append(" align=\"" + align + "\"");
    }
    if (width != null) {
      result.append(" width=\"" + width + "\"");
    }
    if (height != null) {
      result.append(" height=\"" + height + "\"");
    }
    if (alt != null) {
      result.append(" alt=\"" + alt + "\"");
    }
    result.append(">\n");

    // append applet parameters
    for (Iterator i = params.entrySet().iterator(); i.hasNext();) {
      Map.Entry entry = (Map.Entry) i.next();
      result.append("<param name=\"" + entry.getKey() + "\" value=\"" + entry.getValue() + "\"/>");
    }

    result.append("</applet>\n");

    return result.toString();
  }

}